//----------------------------------------------------------------------------//
//- CafArtikelUebersicht.pas -------------------------------------------------//
//----------------------------------------------------------------------------//

//- Beschreibung -------------------------------------------------------------//
// Frame, das eine bersicht von Artikeln darstellt                           //
//----------------------------------------------------------------------------//

unit CafArtikelUebersicht;

//------------------------------------------------------------------------------

interface

//------------------------------------------------------------------------------

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, ActnList, Menus, CaoCheck, CaoArtikelListe, CaoPersonenListe,
  CaoBoersenInfo, CaoFormat;

//------------------------------------------------------------------------------

type
  TFrmArtikelUebersicht = class(TFrame)
    m_LivArtikelUebersicht  : TListView;
    m_AclArtikel            : TActionList;
    m_ActArtikelErfassen    : TAction;
    m_ActArtikelKopieren    : TAction;
    m_ActArtikelBearbeiten  : TAction;
    m_ActArtikelLoeschen    : TAction;
    m_ActEtikettenDrucken   : TAction;
    m_PopArtikel            : TPopupMenu;
    m_MnuEtikettenDrucken   : TMenuItem;
    m_MnuArtikelBearbeiten  : TMenuItem;
    m_MnuArtikelerfassen    : TMenuItem;
    m_MnuArtikelkopieren    : TMenuItem;
    m_MnuArtikelloeschen    : TMenuItem;

    //--------------------------------------------------------------------------

    procedure ActArtikelErfassenExecute    (Sender: TObject);
    procedure ActArtikelKopierenExecute    (Sender: TObject);
    procedure ActEtikettenDruckenExecute   (Sender: TObject);
    procedure ActArtikelBearbeitenExecute  (Sender: TObject);
    procedure ActArtikelLoeschenExecute    (Sender: TObject);
    procedure LivArtikelUebersichtClick    (Sender: TObject);
    procedure LivArtikelUebersichtDblClick (Sender: TObject);

  //----------------------------------------------------------------------------

  private
    m_obCaoCheck       : TCaoCheck;
    m_obSelectedPerson : TCaoPersonen;
    m_obArtikelListe   : TCaoArtikelListe;
    m_obBoersenInfo    : TCaoBoersenInfo;
    m_obFormat         : TCaoFormat;

    //--------------------------------------------------------------------------

    procedure PrvSetEnabled(fEndabled: Boolean);
    procedure PrvInitListView;

  //----------------------------------------------------------------------------

  public
    property  Enabled        : Boolean      write PrvSetEnabled;
    property  SelectedPerson : TCaoPersonen write m_obSelectedPerson;

    //--------------------------------------------------------------------------

    procedure CreateFrame;
    procedure DestroyFrame;
    procedure ShowFrame(var obArtikelListe: TCaoArtikelListe;
                        var obBoersenInfo : TCaoBoersenInfo);
    procedure CloseFrame;
    procedure UpdateFrame;
    procedure SetFocusInListView(obArtikel: TCaoArtikel);

  //----------------------------------------------------------------------------

  end; // TCaFrmArtikelUebersicht

//------------------------------------------------------------------------------

implementation

//------------------------------------------------------------------------------

{$R *.dfm}

//------------------------------------------------------------------------------

uses
  CamConst, CadArtikel, CadEtikettenDrucken, CaoController;

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.ActArtikelErfassenExecute(Sender: TObject);
(*
  Action zum Erfassen eines Artikels
*)
var
  obArtikel          : TCaoArtikel;
  obTempArtikelListe : TCaoArtikelListe;
  iTempModalResult   : Integer;
begin 
  // Diese Temporre Artikelliste wird zum Ausdrucken der aktuell erfassten
  // Etiketten bentigt
  obTempArtikelListe := TCaoArtikelListe.Create;
  // berprfen, ob das Personenobjekt initialisiert ist
  if (m_obSelectedPerson is TCaoPersonen) then begin
    repeat
      // Der ArtikelListe ein Artikel hinzufgen
      obArtikel                 := m_obArtikelListe.CreateArtikel;
      obArtikel.iStatus         := c_iModusNeu;
      obArtikel.fVerkauft       := False;
      obArtikel.pAnbieter       := Pointer(m_obSelectedPerson);
      obArtikel.iBoersenNummer  := m_obBoersenInfo.iBoersenNummer;
      // Wurde der Dialog nicht mit Cancel verlassen, wird der Artikel gespeichert
      iTempModalResult := CaDlgArtikel.ShowModal(obArtikel);
      if iTempModalResult <> mrCancel then begin
        m_obSelectedPerson.obArtikelListe.AddArtikel(obArtikel);
        m_obArtikelListe.SaveToDb;
        obTempArtikelListe.AddArtikel(obArtikel);
      end // if
      // Wurde der Dialog mit Cancel verlassen, so muss das angelegte Objekt
      // wieder zerstrt werden.
      else begin
        m_obArtikelListe.DeleteArtikel(obArtikel);
      end; // else
      PrvInitListView;
    until iTempModalResult <> mrRetry;
    // Wurden neue Artikel erfasst, so wir gefragt, ob Etiketten ausgedruckt
    // werden sollen -> wenn ja werden diese ausgedruckt
    if obTempArtikelListe.Count > 0 then begin
      if MessageDlg(c_sMeldungEtikettenDrucken, mtConfirmation, [mbYes, mbNo], 0)
         = mrYes then begin
        g_obController.EtikettenDrucken(obTempArtikelListe);
      end; // if
    end; // if
    SetFocusInListView(obArtikel);
  end; // if
  obTempArtikelListe.Free(False);
end; // TCaFrmArtikelUebersicht.ActArtikelErfassenExecute

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.ActArtikelKopierenExecute(Sender: TObject);
(*
  Action zum Kopieren eines bestehenden Artikels
*)
var
  obArtikel: TCaoArtikel;
begin
  // berprfen, ob das Personenobjekt initialisiert ist
  if (m_obSelectedPerson is TCaoPersonen) and
      m_obCaoCheck.ListViewItemIsSelected(m_LivArtikelUebersicht) then begin
    // Der ArtikelListe ein Artikel hinzufgen
    obArtikel                  := m_obArtikelListe.CreateArtikel;
    // Das in der Listview selektierte Objekt wird kopiert
    obArtikel.CopyArtikel(TCaoArtikel(m_LivArtikelUebersicht.Selected.Data));
    obArtikel.fVerkauft        := False;
    // Wurde der Dialog nicht mit Cancel verlassen, wird der Artikel gespeichert
    if CaDlgArtikel.ShowModal(obArtikel) <> mrCancel then begin
      m_obSelectedPerson.obArtikelListe.AddArtikel(obArtikel);
      obArtikel.iBoersenNummer := g_obController.obBoersenInfo.iBoersenNummer;
      m_obArtikelListe.SaveToDb;
    end // if
    // Wurde der Dialog mit Cancel verlassen, so muss das angelegte Objekt
    // wieder zerstrt werden.
    else begin
      m_obArtikelListe.DeleteArtikel(obArtikel);
    end; // else
    PrvInitListView;
    SetFocusInListView(obArtikel);
  end; // if
end; // TFrmArtikelUebersicht.ActArtikelKopierenExecute

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.ActEtikettenDruckenExecute(Sender: TObject);
(*
  Action zum Drucken von Etiketten, die anschliessend an den Artikeln befestigt
  werden knnen
*)
var
  obTempArtikelListe: TCaoArtikelListe;
begin
  obTempArtikelListe := TCaoArtikelListe.Create;
  if m_obCaoCheck.ListViewItemIsSelected(m_LivArtikelUebersicht) then begin
    obTempArtikelListe.AddArtikel(TCaoArtikel
                                       (m_LivArtikelUebersicht.Selected.Data));
    g_obController.EtikettenDrucken(obTempArtikelListe);
  end; // if
  obTempArtikelListe.Free(False);
end; // TFrmArtikelUebersicht.ActEtikettenDruckenExecute

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.ActArtikelBearbeitenExecute(Sender: TObject);
(*
  Action zum Bearbeiten bestehender Artikel
*)
var
  obArtikel: TCaoArtikel;
begin
  if m_obCaoCheck.ListViewItemIsSelected(m_LivArtikelUebersicht) then begin
    obArtikel := TCaoArtikel(m_LivArtikelUebersicht.Selected.Data);
    obArtikel.iStatus := c_iModusEdit;
    CaDlgArtikel.ShowModal(obArtikel);
    m_obArtikelListe.SaveToDb;
    PrvInitListView;
    SetFocusInListView(obArtikel);
  end; // if
end; // ActArtikelBearbeitenExecute

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.ActArtikelLoeschenExecute(Sender: TObject);
(*
  Action zum lschen bestehender Artikel
*)
var
  obDeleteArtikel: TCaoArtikel;
begin
  if m_obCaoCheck.ListViewItemIsSelected(m_LivArtikelUebersicht) then begin
    obDeleteArtikel := TCaoArtikel(m_LivArtikelUebersicht.Selected.Data);
    // Entfernen des Artikels aus der ArtikelListe der Person
    m_obSelectedPerson.obArtikelListe.RemoveArtikel(obDeleteArtikel);
    // Das Artikelobjekt als gelscht markieren
    obDeleteArtikel.iStatus := c_iModusDelete;
    m_obArtikelListe.SaveToDb;
    PrvInitListView;
  end; // if
end; // TFrmArtikelUebersicht.ActArtikelLoeschenExecute

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.CreateFrame;
begin
  m_obCaoCheck     := TCaoCheck.Create;
  m_obFormat       := TCaoFormat.Create;
end; // TFrmArtikelUebersicht.CreateFrame

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.DestroyFrame; 
begin
  m_obCaoCheck.Free;
  m_obFormat.Free;
end; // TFrmArtikelUebersicht.DestroyFrame

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.PrvSetEnabled(fEndabled: Boolean);
(*
  Hier knnen die Komponenten des Frames Enabled gesetzt werden
*)
begin
  m_LivArtikelUebersicht.Enabled  := fEndabled;
  m_ActArtikelErfassen.Enabled    := fEndabled;
  m_ActArtikelKopieren.Enabled    := fEndabled;
  m_ActArtikelBearbeiten.Enabled  := fEndabled;
  m_ActArtikelLoeschen.Enabled    := fEndabled;
  m_ActEtikettenDrucken.Enabled   := fEndabled;
end; // TFrmArtikelUebersicht.PrvSetEnabled

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.PrvInitListView;
(*
  Fllt die Listview mit Daten ab
*)
var
  iIndex        : Integer;
  obListViewItem: TListItem;
  sVerkauft     : string;
begin
  m_LivArtikelUebersicht.Items.Clear;
  // Prfen, ob eine Person selektiert wurde
  if m_obSelectedPerson <> nil then begin
    for iIndex := 0 to m_obSelectedPerson.obArtikelListe.Count -1 do begin
      with m_obSelectedPerson.obArtikelListe do begin
        obListViewItem         := m_LivArtikelUebersicht.Items.Add;
        obListViewItem.Data    := Items[iIndex];
        obListViewItem.Caption := Items[iIndex].sBeschreibung;
        obListViewItem.SubItems.Add(m_obFormat.FormatCurrency(Items[iIndex].dPreis));
        if Items[iIndex].fVerkauft then begin
          sVerkauft := 'Ja';
        end // if
        else begin
          sVerkauft := 'Nein';
        end; // else
        obListViewItem.SubItems.Add(sVerkauft);
      end; // with
    end; // for
  end; // if
end; // TFrmArtikelUebersicht.PrvInitListView

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.CloseFrame;
begin

end; // TFrmArtikelUebersicht.CloseFrame

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.ShowFrame(var obArtikelListe: TCaoArtikelListe;
                                          var obBoersenInfo : TCaoBoersenInfo);
begin
  m_obBoersenInfo  := obBoersenInfo;
  m_obArtikelListe := obArtikelListe;
end; // TFrmArtikelUebersicht.ShowFrame

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.UpdateFrame;
begin
  PrvInitListView;
end; // TFrmArtikelUebersicht.UpdateFrame

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.LivArtikelUebersichtClick(Sender: TObject);
begin
 ActArtikelBearbeitenExecute(Sender);
end; // TFrmArtikelUebersicht.LivArtikelUebersichtClick

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.LivArtikelUebersichtDblClick(Sender: TObject);
begin
  ActArtikelBearbeitenExecute(Sender);
end; // TFrmArtikelUebersicht.LivArtikelUebersichtDblClick

//------------------------------------------------------------------------------

procedure TFrmArtikelUebersicht.SetFocusInListView(obArtikel: TCaoArtikel);
(*
  Setzt den Fokus in der Listview auf das Objekt, das per Parameter mitgegeben
  wurde
*)
var
  iIndex: Integer;
begin
  for iIndex := 0 to m_LivArtikelUebersicht.Items.Count -1 do begin
    if m_LivArtikelUebersicht.Items[iIndex].Data = Pointer(obArtikel) then begin
      m_LivArtikelUebersicht.SetFocus;
      m_LivArtikelUebersicht.Items[iIndex].Selected := True;
      Break;
    end; // if
  end; // for
end; // TFrmArtikelUebersicht.SetFocusInListView

//------------------------------------------------------------------------------

end. // CafArtikelUebersicht
